/*
 * Decompiled with CFR 0.152.
 */
package org.spongepowered.vanilla.installer;

import com.google.gson.Gson;
import com.google.gson.stream.JsonReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.lang.reflect.Type;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.AccessDeniedException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.StringJoiner;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.zip.ZipEntry;
import net.minecraftforge.fart.api.Renamer;
import net.minecraftforge.fart.api.SignatureStripperConfig;
import net.minecraftforge.fart.api.SourceFixerConfig;
import net.minecraftforge.fart.api.Transformer;
import net.minecraftforge.srgutils.IMappingFile;
import org.spongepowered.bootstrap.forge.VanillaBootstrap;
import org.spongepowered.libs.LibraryManager;
import org.spongepowered.libs.LibraryUtils;
import org.spongepowered.vanilla.installer.Installer;
import org.spongepowered.vanilla.installer.LauncherCommandLine;
import org.spongepowered.vanilla.installer.library.TinyLogger;
import org.spongepowered.vanilla.installer.model.GroupArtifactVersion;
import org.spongepowered.vanilla.installer.model.mojang.BundleElement;
import org.spongepowered.vanilla.installer.model.mojang.BundlerMetadata;
import org.spongepowered.vanilla.installer.model.mojang.FormatVersion;
import org.spongepowered.vanilla.installer.model.mojang.Version;
import org.spongepowered.vanilla.installer.model.mojang.VersionManifest;
import org.tinylog.Logger;

public final class InstallerMain {
    private static final String COLLECTION_BOOTSTRAP = "bootstrap";
    private static final String COLLECTION_MAIN = "main";
    private static final int MAX_TRIES = 2;
    private final Installer installer;
    private final boolean isolated;

    public InstallerMain(String[] args, boolean isolated) throws Exception {
        LauncherCommandLine.configure(args);
        this.installer = new Installer(LauncherCommandLine.installerDirectory);
        this.isolated = isolated;
    }

    public static void main(String[] args) throws Exception {
        new InstallerMain(args, true).run();
    }

    public void run() throws Exception {
        try {
            this.downloadAndRun();
        }
        catch (Exception ex) {
            Logger.error((Throwable)ex, "Failed to download Sponge libraries and/or Minecraft");
            throw ex;
        }
        finally {
            this.installer.getLibraryManager().finishedProcessing();
        }
    }

    private void downloadAndRun() throws Exception {
        ServerAndLibraries remappedMinecraftJar = null;
        Version mcVersion = null;
        try {
            mcVersion = this.downloadMinecraftManifest();
        }
        catch (IOException ex) {
            remappedMinecraftJar = this.recoverFromMinecraftDownloadError(ex);
        }
        LibraryManager libraryManager = this.installer.getLibraryManager();
        try {
            if (mcVersion != null) {
                CompletableFuture<Path> mappingsFuture = this.downloadMappings(mcVersion);
                CompletableFuture<Path> originalMcFuture = this.downloadMinecraft(mcVersion);
                CompletionStage extractedFuture = originalMcFuture.thenApplyAsync(this::extractBundle, (Executor)libraryManager.preparationWorker());
                CompletionStage remappedMinecraftJarFuture = mappingsFuture.thenCombineAsync(extractedFuture, (mappings, minecraft) -> {
                    try {
                        return this.remapMinecraft((ServerAndLibraries)minecraft, (Path)mappings);
                    }
                    catch (IOException ex) {
                        throw new UncheckedIOException(ex);
                    }
                }, (Executor)libraryManager.preparationWorker());
                remappedMinecraftJar = (ServerAndLibraries)((CompletableFuture)remappedMinecraftJarFuture).get();
            }
        }
        catch (ExecutionException ex) {
            Throwable cause = ex.getCause();
            remappedMinecraftJar = this.recoverFromMinecraftDownloadError(cause instanceof Exception ? (Exception)cause : ex);
        }
        assert (remappedMinecraftJar != null);
        libraryManager.validate();
        libraryManager.addLibrary(COLLECTION_MAIN, new LibraryManager.Library("minecraft", remappedMinecraftJar.server()));
        for (Map.Entry<GroupArtifactVersion, Path> entry : remappedMinecraftJar.libraries().entrySet()) {
            GroupArtifactVersion artifact = entry.getKey();
            Path path = entry.getValue();
            libraryManager.addLibrary(COLLECTION_BOOTSTRAP, new LibraryManager.Library(artifact.toString(), path));
        }
        if (!this.isolated) {
            Path jacocoJar = null;
            try {
                Class<?> jacocoClass = this.getClass().getClassLoader().loadClass("org.jacoco.core.JaCoCo");
                jacocoJar = Path.of(jacocoClass.getProtectionDomain().getCodeSource().getLocation().toURI());
            }
            catch (Exception jacocoClass) {
                // empty catch block
            }
            if (jacocoJar != null && jacocoJar.getFileName().toString().endsWith(".jar")) {
                Logger.info("JaCoCo core has been detected. Custom instrumentation will be enabled.");
                libraryManager.addLibrary(COLLECTION_BOOTSTRAP, new LibraryManager.Library("jacoco-core", jacocoJar));
            }
        }
        libraryManager.finishedProcessing();
        Logger.info("Environment has been verified.");
        HashSet seenLibs = new HashSet();
        Path[] bootLibs = (Path[])libraryManager.getAll(COLLECTION_BOOTSTRAP).stream().peek(lib -> seenLibs.add(lib.name())).map(LibraryManager.Library::file).toArray(Path[]::new);
        Path[] gameLibs = (Path[])libraryManager.getAll(COLLECTION_MAIN).stream().filter(lib -> !seenLibs.contains(lib.name())).map(LibraryManager.Library::file).toArray(Path[]::new);
        URL rootJar = InstallerMain.class.getProtectionDomain().getCodeSource().getLocation();
        URI fsURI = new URI("jar:" + String.valueOf(rootJar));
        System.setProperty("sponge.rootJarFS", fsURI.toString());
        FileSystem fs = FileSystems.newFileSystem(fsURI, Map.of());
        Path spongeBoot = InstallerMain.newJarInJar(fs.getPath("jars", "spongevanilla-boot.jar"));
        String launchTarget = LauncherCommandLine.launchTarget;
        if (launchTarget == null) {
            Path manifestFile = fs.getPath("META-INF", "MANIFEST.MF");
            try (InputStream stream = Files.newInputStream(manifestFile, new OpenOption[0]);){
                Manifest manifest = new Manifest(stream);
                launchTarget = manifest.getMainAttributes().getValue("Launch-Target");
            }
        }
        StringJoiner resourcesEnv = new StringJoiner(File.pathSeparator);
        for (Path lib2 : gameLibs) {
            resourcesEnv.add(lib2.toAbsolutePath().toString());
        }
        System.setProperty("sponge.resources", resourcesEnv.toString());
        ArrayList<String> gameArgs = new ArrayList<String>(LauncherCommandLine.remainingArgs);
        gameArgs.add("--launchTarget");
        gameArgs.add(launchTarget);
        Collections.addAll(gameArgs, this.installer.getConfig().args().split(" "));
        this.bootstrap(bootLibs, spongeBoot, gameArgs.toArray(new String[0]));
    }

    private static Path newJarInJar(Path jar) {
        try {
            URI jij = new URI("jij:" + jar.toAbsolutePath().toUri().getRawSchemeSpecificPart()).normalize();
            Map<String, Path> env = Map.of("packagePath", jar);
            FileSystem jijFS = FileSystems.newFileSystem(jij, env);
            return jijFS.getPath("/", new String[0]);
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private <T extends Throwable> ServerAndLibraries recoverFromMinecraftDownloadError(T ex) throws T {
        Path expectedUnpacked = this.expectedMinecraftLocation("1.21.1");
        Path expectedRemapped = this.expectedRemappedLocation(expectedUnpacked);
        if (Files.exists(expectedRemapped, new LinkOption[0])) {
            Logger.warn(ex, "Failed to download and remap Minecraft. An existing jar exists, so we will attempt to use that instead.");
            return this.extractBundle(this.expectedBundleLocation(expectedUnpacked));
        }
        throw ex;
    }

    private void bootstrap(Path[] bootLibs, Path spongeBoot, String[] args) throws Exception {
        ArrayList<Path[]> classpath = new ArrayList<Path[]>();
        for (Path lib : bootLibs) {
            classpath.add(new Path[]{lib});
        }
        classpath.add(new Path[]{spongeBoot});
        try {
            new VanillaBootstrap(args).boot(classpath, this.isolated);
        }
        catch (Exception ex) {
            Logger.error((Throwable)ex, "Failed to invoke bootstrap due to an error");
            throw ex;
        }
    }

    private Version downloadMinecraftManifest() throws Exception {
        Version version;
        Logger.info("Downloading the Minecraft versions manifest...");
        VersionManifest.Version foundVersionManifest = null;
        Gson gson = new Gson();
        URLConnection conn = new URI("https://launchermeta.mojang.com/mc/game/version_manifest.json").toURL().openConnection();
        conn.setConnectTimeout(5000);
        try (JsonReader reader = new JsonReader(new InputStreamReader(conn.getInputStream()));){
            VersionManifest manifest = (VersionManifest)gson.fromJson(reader, (Type)((Object)VersionManifest.class));
            for (VersionManifest.Version version2 : manifest.versions()) {
                if (!"1.21.1".equals(version2.id())) continue;
                foundVersionManifest = version2;
                break;
            }
        }
        if (foundVersionManifest == null) {
            throw new IOException(String.format("Failed to find version manifest for '%s'!", "1.21.1"));
        }
        try (JsonReader reader = new JsonReader(new InputStreamReader(foundVersionManifest.url().openStream()));){
            version = (Version)gson.fromJson(reader, (Type)((Object)Version.class));
        }
        if (version == null) {
            throw new IOException(String.format("Failed to download version information for '%s'!", "1.21.1"));
        }
        return version;
    }

    private Path expectedMinecraftLocation(String version) {
        return this.installer.getLibraryManager().getRootDirectory().resolve("net/minecraft").resolve(version).resolve("minecraft_server.jar");
    }

    private Path expectedRemappedLocation(Path originalLocation) {
        return originalLocation.resolveSibling("minecraft_server-remapped.jar");
    }

    private Path expectedBundleLocation(Path originalLocation) {
        return originalLocation.resolveSibling("minecraft_server-bundle.jar");
    }

    private CompletableFuture<Path> downloadMinecraft(Version version) {
        return LibraryUtils.asyncFailableFuture(() -> {
            Path downloadTarget = this.expectedBundleLocation(this.expectedMinecraftLocation(version.id()));
            Version.Downloads.Download server = version.downloads().server();
            if (Files.notExists(downloadTarget, new LinkOption[0])) {
                if (!this.installer.getConfig().autoDownloadLibraries()) {
                    throw new IOException(String.format("The Minecraft jar is not located at '%s' and downloading it has been turned off.", downloadTarget));
                }
                LibraryUtils.downloadAndVerifyDigest(TinyLogger.INSTANCE, server.url(), downloadTarget, "SHA-1", server.sha1());
            } else if (this.installer.getConfig().checkLibraryHashes()) {
                Logger.info("Detected existing Minecraft Server jar, verifying hashes...");
                if (LibraryUtils.validateDigest("SHA-1", server.sha1(), downloadTarget)) {
                    Logger.info("Minecraft Server jar verified!");
                } else {
                    Logger.error("Checksum verification failed: Expected {}. Deleting cached Minecraft Server jar...", server.sha1());
                    Files.delete(downloadTarget);
                    LibraryUtils.downloadAndVerifyDigest(TinyLogger.INSTANCE, server.url(), downloadTarget, "SHA-1", server.sha1());
                }
            } else {
                Logger.info("Detected existing Minecraft jar. Skipping hash check as that is turned off...");
            }
            return downloadTarget;
        }, this.installer.getLibraryManager().preparationWorker());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private ServerAndLibraries extractBundle(Path bundleJar) {
        Path serverDestination = this.expectedMinecraftLocation("1.21.1");
        try (JarFile bundle = new JarFile(bundleJar.toFile());){
            Optional<BundlerMetadata> metaOpt = BundlerMetadata.read(bundle);
            if (metaOpt.isEmpty()) {
                ServerAndLibraries serverAndLibraries = new ServerAndLibraries(bundleJar, Map.of());
                return serverAndLibraries;
            }
            BundlerMetadata md = metaOpt.get();
            if (!md.version().equals(new FormatVersion(1, 0))) {
                Logger.warn("Read bundler metadata from server jar with version {}, but we only support 1.0", md.version());
            }
            boolean serverExtractionNeeded = true;
            BundleElement server = md.server();
            if (Files.exists(serverDestination, new LinkOption[0]) && LibraryUtils.validateDigest("SHA-256", server.sha256(), serverDestination)) {
                serverExtractionNeeded = false;
            }
            if (serverExtractionNeeded) {
                ZipEntry serverEntry = bundle.getEntry(server.path());
                try (InputStream in = bundle.getInputStream(serverEntry);){
                    LibraryUtils.transferAndVerifyDigest(TinyLogger.INSTANCE, in, serverDestination, "SHA-256", server.sha256());
                }
            }
            Path libsDir = this.installer.getLibraryManager().getRootDirectory();
            HashMap<GroupArtifactVersion, Path> libs = new HashMap<GroupArtifactVersion, Path>();
            for (BundleElement library : md.libraries()) {
                GroupArtifactVersion gav = GroupArtifactVersion.parse(library.id());
                Path destination = gav.resolve(libsDir).resolve(gav.artifact() + "-" + gav.version() + (String)(gav.classifier() == null ? "" : "-" + gav.classifier()) + ".jar");
                if (Files.exists(destination, new LinkOption[0]) && LibraryUtils.validateDigest("SHA-256", library.sha256(), destination)) {
                    libs.put(gav, destination);
                    continue;
                }
                ZipEntry entry = bundle.getEntry(library.path());
                InputStream in = bundle.getInputStream(entry);
                try {
                    LibraryUtils.transferAndVerifyDigest(TinyLogger.INSTANCE, in, destination, "SHA-256", library.sha256());
                    libs.put(gav, destination);
                }
                finally {
                    if (in == null) continue;
                    in.close();
                }
            }
            ServerAndLibraries serverAndLibraries = new ServerAndLibraries(serverDestination, libs);
            return serverAndLibraries;
        }
        catch (IOException | NoSuchAlgorithmException ex) {
            Logger.error((Throwable)ex, "Failed to extract bundle from {}", bundleJar);
            throw new RuntimeException(ex);
        }
    }

    private CompletableFuture<Path> downloadMappings(Version version) {
        return LibraryUtils.asyncFailableFuture(() -> {
            Logger.info("Setting up names for Minecraft {}", "1.21.1");
            Path downloadTarget = this.installer.getLibraryManager().getRootDirectory().resolve("net/minecraft/mappings").resolve("1.21.1").resolve("server.txt");
            Version.Downloads.Download mappings = version.downloads().server_mappings();
            if (mappings == null) {
                throw new IOException(String.format("Mappings were not included in version manifest for %s", "1.21.1"));
            }
            boolean checkHashes = this.installer.getConfig().checkLibraryHashes();
            if (Files.exists(downloadTarget, new LinkOption[0])) {
                if (checkHashes) {
                    Logger.info("Detected existing mappings, verifying hashes...");
                    if (LibraryUtils.validateDigest("SHA-1", mappings.sha1(), downloadTarget)) {
                        Logger.info("Mappings verified!");
                        return downloadTarget;
                    }
                    Logger.error("Checksum verification failed: Expected {}. Deleting cached server mappings file...", mappings.sha1());
                    Files.delete(downloadTarget);
                } else {
                    return downloadTarget;
                }
            }
            if (this.installer.getConfig().autoDownloadLibraries()) {
                if (checkHashes) {
                    LibraryUtils.downloadAndVerifyDigest(TinyLogger.INSTANCE, mappings.url(), downloadTarget, "SHA-1", mappings.sha1());
                } else {
                    LibraryUtils.download(TinyLogger.INSTANCE, mappings.url(), downloadTarget, false);
                }
            } else {
                throw new IOException(String.format("Mappings were not located at '%s' and downloading them has been turned off.", downloadTarget));
            }
            return downloadTarget;
        }, this.installer.getLibraryManager().preparationWorker());
    }

    private ServerAndLibraries remapMinecraft(ServerAndLibraries minecraft, Path serverMappings) throws IOException {
        Logger.info("Checking if we need to remap Minecraft...");
        Path outputJar = this.expectedRemappedLocation(minecraft.server());
        Path tempOutput = outputJar.resolveSibling("minecraft_server_remapped.jar.tmp");
        if (Files.exists(outputJar, new LinkOption[0])) {
            Logger.info("Remapped Minecraft detected, skipping...");
            return minecraft.server(outputJar);
        }
        Logger.info("Remapping Minecraft. This may take a while...");
        IMappingFile mappings = IMappingFile.load(serverMappings.toFile()).reverse();
        Renamer.Builder renamerBuilder = Renamer.builder().add(Transformer.parameterAnnotationFixerFactory()).add(ctx -> {
            final Transformer backing = Transformer.renamerFactory(mappings, false).create(ctx);
            return new Transformer(){

                @Override
                public Transformer.ClassEntry process(Transformer.ClassEntry entry) {
                    String name = entry.getName();
                    if (name.startsWith("it/unimi") || name.startsWith("com/google") || name.startsWith("com/mojang/datafixers") || name.startsWith("com/mojang/brigadier") || name.startsWith("org/apache")) {
                        return entry;
                    }
                    return backing.process(entry);
                }

                @Override
                public Transformer.ManifestEntry process(Transformer.ManifestEntry entry) {
                    return backing.process(entry);
                }

                @Override
                public Transformer.ResourceEntry process(Transformer.ResourceEntry entry) {
                    return backing.process(entry);
                }

                @Override
                public Collection<? extends Transformer.Entry> getExtras() {
                    return backing.getExtras();
                }
            };
        }).add(Transformer.recordFixerFactory()).add(Transformer.parameterAnnotationFixerFactory()).add(Transformer.sourceFixerFactory(SourceFixerConfig.JAVA)).add(Transformer.signatureStripperFactory(SignatureStripperConfig.ALL)).logger(Logger::debug);
        try (Renamer ren = renamerBuilder.build();){
            ren.run(minecraft.server.toFile(), tempOutput.toFile());
        }
        try {
            Files.move(tempOutput, outputJar, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (AccessDeniedException ex) {
            for (int tries = 0; tries < 2; ++tries) {
                try {
                    Thread.sleep(5 * tries);
                    Files.move(tempOutput, outputJar, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
                    continue;
                }
                catch (AccessDeniedException ex2) {
                    if (tries != 1) continue;
                    throw ex;
                }
                catch (InterruptedException exInterrupt) {
                    Thread.currentThread().interrupt();
                    throw ex;
                }
            }
        }
        return minecraft.server(outputJar);
    }

    record ServerAndLibraries(Path server, Map<GroupArtifactVersion, Path> libraries) {
        ServerAndLibraries {
            libraries = Map.copyOf(libraries);
        }

        public ServerAndLibraries server(Path server) {
            return new ServerAndLibraries(server, this.libraries);
        }
    }
}

