/*
 * Decompiled with CFR 0.152.
 */
package org.spongepowered.libs;

import com.google.gson.Gson;
import com.google.gson.stream.JsonReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.spongepowered.libs.LibraryUtils;
import org.spongepowered.libs.Logger;
import org.spongepowered.libs.model.Libraries;
import org.spongepowered.libs.model.SonatypeResponse;

public final class LibraryManager {
    public static final String SPONGE_NEXUS_DOWNLOAD_URL = "https://repo.spongepowered.org/service/rest/v1/search/assets?sha512=%s&maven.groupId=%s&maven.artifactId=%s&maven.baseVersion=%s&maven.extension=jar";
    private static final List<String> PREFERRED_REPOSITORY_ORDER = List.of("maven-central", "minecraft-proxy", "maven-releases", "maven-snapshots", "google-proxy", "forge-proxy", "neoforge-releases", "neoforge-snapshots", "fabric-proxy");
    private final Logger logger;
    private final boolean checkLibraryHashes;
    private final Path rootDirectory;
    private final URL librariesUrl;
    private final Map<String, Set<Library>> libraries;
    private final ExecutorService preparationWorker;
    private final Gson gson;

    public LibraryManager(Logger logger, boolean checkLibraryHashes, Path rootDirectory, URL librariesUrl) {
        this.logger = Objects.requireNonNull(logger, "logger");
        this.checkLibraryHashes = checkLibraryHashes;
        this.rootDirectory = Objects.requireNonNull(rootDirectory, "rootDirectory");
        this.librariesUrl = Objects.requireNonNull(librariesUrl, "librariesUrl");
        this.libraries = new LinkedHashMap<String, Set<Library>>();
        int availableCpus = Runtime.getRuntime().availableProcessors();
        this.preparationWorker = new ThreadPoolExecutor(Math.min(Math.max(4, availableCpus * 2), 64), Integer.MAX_VALUE, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>());
        this.gson = new Gson();
    }

    public Path getRootDirectory() {
        return this.rootDirectory;
    }

    public Set<Library> getAll(String collection) {
        return Collections.unmodifiableSet(this.libraries.getOrDefault(collection, Collections.emptySet()));
    }

    public void addLibrary(String set, Library library) {
        this.libraries.computeIfAbsent(set, key -> Collections.synchronizedSet(new LinkedHashSet())).add(library);
    }

    public void validate() throws Exception {
        Libraries dependencies;
        this.logger.info("Scanning and verifying libraries in '{}'. Please wait, this may take a moment...", this.rootDirectory);
        try (JsonReader reader = new JsonReader((Reader)new InputStreamReader(this.librariesUrl.openStream(), StandardCharsets.UTF_8));){
            dependencies = (Libraries)this.gson.fromJson(reader, Libraries.class);
        }
        HashMap<String, Set<Library>> downloadedDeps = new HashMap<String, Set<Library>>();
        HashMap<String, CompletableFuture<Path>> operations = new HashMap<String, CompletableFuture<Path>>();
        ConcurrentHashMap.KeySetView failures = ConcurrentHashMap.newKeySet();
        for (Map.Entry<String, List<Libraries.Dependency>> entry : dependencies.dependencies().entrySet()) {
            downloadedDeps.put(entry.getKey(), this.scheduleDownloads(entry.getValue(), operations, failures));
        }
        ((CompletableFuture)CompletableFuture.allOf(operations.values().toArray(new CompletableFuture[0])).handle((result, err) -> {
            if (err != null) {
                failures.add(err.getMessage());
                this.logger.error("Failed to download library", err);
            }
            return result;
        })).join();
        if (!failures.isEmpty()) {
            this.logger.error("Failed to download some libraries:", new Object[0]);
            for (String message : failures) {
                this.logger.error(message, new Object[0]);
            }
            System.exit(-1);
        }
        this.libraries.putAll(downloadedDeps);
    }

    private Set<Library> scheduleDownloads(List<Libraries.Dependency> dependencies, Map<String, CompletableFuture<Path>> operations, Set<String> failures) {
        Set<Library> downloadedDeps = Collections.synchronizedSet(new LinkedHashSet(dependencies.size()));
        for (Libraries.Dependency dep : dependencies) {
            operations.computeIfAbsent(LibraryManager.getId(dep), key -> LibraryUtils.asyncFailableFuture(() -> {
                URL requestUrl;
                SonatypeResponse response;
                String groupPath = dep.group().replace(".", "/");
                Path depDirectory = this.rootDirectory.resolve(groupPath).resolve(dep.module()).resolve(dep.version());
                Files.createDirectories(depDirectory, new FileAttribute[0]);
                Path depFile = depDirectory.resolve(dep.module() + "-" + dep.version() + ".jar");
                boolean checkHashes = this.checkLibraryHashes;
                if (Files.exists(depFile, new LinkOption[0])) {
                    if (!checkHashes) {
                        this.logger.info("Detected existing '{}', skipping hash checks...", depFile);
                        return depFile;
                    }
                    if (LibraryUtils.validateDigest("SHA-512", dep.sha512(), depFile)) {
                        this.logger.debug("'{}' verified!", depFile);
                        return depFile;
                    }
                    this.logger.error("Checksum verification failed: Expected {}. Deleting cached '{}'...", dep.sha512(), depFile);
                    Files.delete(depFile);
                }
                if ((response = this.getResponseFor(this.gson, requestUrl = new URI(String.format(SPONGE_NEXUS_DOWNLOAD_URL, dep.sha512(), dep.group(), dep.module(), dep.version())).toURL())).items().isEmpty()) {
                    failures.add("No data received from '" + String.valueOf(requestUrl) + "'!");
                    return null;
                }
                SonatypeResponse.Item item = response.items().stream().min(Comparator.comparingInt(i -> {
                    if (!PREFERRED_REPOSITORY_ORDER.contains(i.repository())) {
                        return Integer.MAX_VALUE;
                    }
                    return PREFERRED_REPOSITORY_ORDER.indexOf(i.repository());
                })).get();
                if (checkHashes) {
                    LibraryUtils.downloadAndVerifyDigest(this.logger, item.downloadUrl(), depFile, "SHA-512", item.checksum().sha512());
                } else {
                    LibraryUtils.download(this.logger, item.downloadUrl(), depFile, true);
                }
                return depFile;
            }, this.preparationWorker)).whenComplete((res, err) -> {
                if (res != null) {
                    downloadedDeps.add(new Library(LibraryManager.getId(dep), (Path)res));
                }
            });
        }
        return downloadedDeps;
    }

    private SonatypeResponse getResponseFor(Gson gson, URL requestUrl) throws IOException {
        HttpURLConnection connection = (HttpURLConnection)requestUrl.openConnection();
        connection.setRequestMethod("GET");
        connection.setRequestProperty("Content-Type", "application/json");
        connection.setRequestProperty("User-Agent", "Sponge-Downloader");
        connection.connect();
        try (JsonReader reader = new JsonReader((Reader)new InputStreamReader(connection.getInputStream()));){
            SonatypeResponse sonatypeResponse = (SonatypeResponse)gson.fromJson(reader, SonatypeResponse.class);
            return sonatypeResponse;
        }
    }

    public ExecutorService preparationWorker() {
        return this.preparationWorker;
    }

    public void finishedProcessing() {
        boolean successful;
        if (this.preparationWorker.isTerminated()) {
            return;
        }
        this.preparationWorker.shutdown();
        try {
            successful = this.preparationWorker.awaitTermination(10L, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            successful = false;
        }
        if (!successful) {
            this.logger.warn("Failed to shut down library preparation pool in 10 seconds, forcing shutdown now.", new Object[0]);
            this.preparationWorker.shutdownNow();
        }
    }

    private static String getId(Libraries.Dependency dep) {
        return dep.group() + ":" + dep.module() + ":" + dep.version();
    }

    public record Library(String name, Path file) {
    }
}

